#include "..\huc.h"


/* VRAM locations */

#define VRAM_SELECT     0x3400

#define VRAM_WHITE1     0x3000
#define VRAM_WHITE2     0x3010
#define VRAM_WHITE3     0x3020
#define VRAM_WHITE4     0x3030

#define VRAM_BLACK1     0x3040
#define VRAM_BLACK2     0x3050
#define VRAM_BLACK3     0x3060
#define VRAM_BLACK4     0x3070

#define VRAM_EMPTY1     0x3080
#define VRAM_EMPTY2     0x3090
#define VRAM_EMPTY3     0x3100
#define VRAM_EMPTY4     0x3110


/* miscellaneous constants */

#define BGPAL           0
#define SPRPAL          0

#define X_BOARD_OFFSET  8
#define Y_BOARD_OFFSET  8
#define X_SPR_OFFSET    64
#define Y_SPR_OFFSET    64
#define CHARS_PER_LINE  64

#define LOOK_NEUT   0
#define LOOK_LEFT   1
#define LOOK_RIGHT  2
#define LOOK_UP     1
#define LOOK_DOWN   2

#define PLAYER_WHITE  1
#define PLAYER_BLACK  2


/* include graphics */

#incchr(white1, "squares.pcx", 0,16, 1, 1);
#incchr(white2, "squares.pcx", 8,16, 1, 1);
#incchr(white3, "squares.pcx", 0,24, 1, 1);
#incchr(white4, "squares.pcx", 8,24, 1, 1);

#incchr(black1, "squares.pcx",16, 0, 1, 1);
#incchr(black2, "squares.pcx",24, 0, 1, 1);
#incchr(black3, "squares.pcx",16, 8, 1, 1);
#incchr(black4, "squares.pcx",24, 8, 1, 1);

#incchr(empty1, "squares.pcx",16,16, 1, 1);
#incchr(empty2, "squares.pcx",24,16, 1, 1);
#incchr(empty3, "squares.pcx",16,24, 1, 1);
#incchr(empty4, "squares.pcx",24,24, 1, 1);

#incspr(select,"squares.pcx", 0, 0, 1, 1);

#incpal(colors, "squares.pcx", 0, 1);



/* GLOBALS */

/* attributes for each square on board */

char position[64];
char changed[64];
char priority[64];

/* scores */

char total1;
char total2;

/* selector position */

char ptrx;
char ptry;

/* Misc */

char selected;
char player;
char computer;

char maxprio;
char maxpos;


/* --------- START FLIPIT ---------- */

/* display score */

dispscore()
{
char i;

    put_string("white", 0x140);
    put_string("black", 0x159);

    put_string("score", 0x2C0);
    put_string("score", 0x2D9);

    if (player == PLAYER_WHITE)
    {
        put_string("players ", 0x1C0);
        put_string(" turn   ", 0x200);
        put_string("        ", 0x1D9);
        put_string("        ", 0x219);
    }
    else if (player == PLAYER_BLACK)
    {
        put_string("        ", 0x1C0);
        put_string("        ", 0x200);

	if (computer == 0)
	{
            put_string("players ", 0x1D9);
            put_string(" turn   ", 0x219);
	}
	else
	{
            put_string("thinking", 0x1D9);
	}
    }
    else
    {
        put_string("        ", 0x1C0);
        put_string("        ", 0x200);
        put_string("        ", 0x1D9);
        put_string("        ", 0x219);
    }

    total1 = 0;
    total2 = 0;

    for (i=0; i<64; i++)
    {
        switch (position[i])
	{
        case PLAYER_WHITE:
            total1++;
            break;

	case PLAYER_BLACK:
	    total2++;
	    break;
	}
    }

    put_number(total1, 2, 0x341);
    put_number(total2, 2, 0x35A);
}


/* position player's selection pointer */

ptr_move()
{
    spr_set(0);

    if (ptrx < 8)
        spr_x(X_SPR_OFFSET + (ptrx<<4) );
    else
        spr_x(0);

    if (ptry < 8)
        spr_y(Y_SPR_OFFSET + (ptry<<4) );
    else
        spr_y(0);

    spr_pattern(VRAM_SELECT);
    spr_ctrl(SIZE_MAS|FLIP_MAS, SZ_16x16|NO_FLIP);
    spr_pal(SPRPAL);
    spr_pri(1);
}


/* display an individual square */

display(pos)
char pos;
{
char x,y,temp;
int bataddr;

    x = pos & 7;
    y = pos >> 3;

    bataddr = ((y<<1)+Y_BOARD_OFFSET) * CHARS_PER_LINE + (x<<1) + X_BOARD_OFFSET;

    temp = position[pos];

    if (temp == PLAYER_WHITE)
    {
        put_raw( (BGPAL<<12)|(VRAM_WHITE1>>4), bataddr++);
        put_raw( (BGPAL<<12)|(VRAM_WHITE2>>4), bataddr);

	bataddr += CHARS_PER_LINE-1;

        put_raw( (BGPAL<<12)|(VRAM_WHITE3>>4), bataddr++);
        put_raw( (BGPAL<<12)|(VRAM_WHITE4>>4), bataddr);
    }

    else if (temp == PLAYER_BLACK)
    {
        put_raw( (BGPAL<<12)|(VRAM_BLACK1>>4), bataddr++);
        put_raw( (BGPAL<<12)|(VRAM_BLACK2>>4), bataddr);

	bataddr += CHARS_PER_LINE-1;

        put_raw( (BGPAL<<12)|(VRAM_BLACK3>>4), bataddr++);
        put_raw( (BGPAL<<12)|(VRAM_BLACK4>>4), bataddr);
    }

    else
    {
        put_raw( (BGPAL<<12)|(VRAM_EMPTY1>>4), bataddr++);
        put_raw( (BGPAL<<12)|(VRAM_EMPTY2>>4), bataddr);

	bataddr += CHARS_PER_LINE-1;

        put_raw( (BGPAL<<12)|(VRAM_EMPTY3>>4), bataddr++);
        put_raw( (BGPAL<<12)|(VRAM_EMPTY4>>4), bataddr);
    }
}


displayall()
{
char i;

    for (i=0; i<64; i++)
    {
        display(i);
	changed[i] = 0;
    }
}


displaydirty()
{
char i;

    for (i=0; i<64; i++)
    {
        if (changed[i] != 0)
	{
            display(i);
	    changed[i] = 0;
	}
    }
}


disppriority()
{
char i,x,y;
int bataddr;

    for (i=0; i<64; i++)
    {
        y = i >> 3;
	x = i & 7;

	if (priority[i] != 0)
	{
            bataddr = ((y<<1)+Y_BOARD_OFFSET+1) * CHARS_PER_LINE + (x<<1) + X_BOARD_OFFSET;
            put_number(priority[i], 1, bataddr);
	}
    }
}


setpriority()
{
char i,x,y;

    for (i=0; i<64; i++)
    {
        y = i >> 3;
	x = i & 7;
        priority[i] = okmove(x,y);
    }
}


/* return the value of the piece in the (x,y) location */

occupies(x, y)
char x,y;
{
char location;

    location = (y<<3) | x;
    return(position[location]);
}


adjustxy(x,y,dx,dy)
char *x;
char *y;
char dx,dy;
{
    switch (dx)
    {
    case LOOK_LEFT:
        if (*x == 0)
            return(1);

	(*x)--;
	break;

    case LOOK_RIGHT:
	if (*x == 7)
            return(1);

	(*x)++;
	break;
    }

    switch (dy)
    {
    case LOOK_UP:
        if (*y == 0)
            return(1);

	(*y)--;
	break;

    case LOOK_DOWN:
	if (*y == 7)
            return(1);

	(*y)++;
	break;
    }

    return(0);
}


/* count how many pieces can flip in this direction  */
/* given a start point (x,y) and a direction (dx,dy) */

testline(x,y,dx,dy)
char x,y,dx,dy;
{
char otherplay;
char flag;
char tmp;

    tmp = 0;

    if (player == PLAYER_WHITE)
        otherplay = PLAYER_BLACK;
    else
        otherplay = PLAYER_WHITE;

    flag = adjustxy(&x,&y,dx,dy);

    if (flag == 1)
        return(0);

    while ((occupies(x,y) == otherplay) && (flag == 0))
    {
        tmp++;
	flag = adjustxy(&x,&y,dx,dy);
    }

    if (flag == 1)
        return(0);

    if (occupies(x,y) == player)
        return(tmp);

    return(0);
}


/* check how many pieces will flip for this move    */
/* in all directions.  Zero means move is not valid */

okmove(x,y)
char x,y;
{
char captured;

    captured = 0;

    if (occupies(x, y) != 0)
        return(0);

    captured += testline(x,y,LOOK_RIGHT,LOOK_NEUT);	/* right */
    captured += testline(x,y,LOOK_RIGHT,LOOK_DOWN);	/* diag dn-rt */
    captured += testline(x,y,LOOK_NEUT, LOOK_DOWN);	/* down */
    captured += testline(x,y,LOOK_LEFT, LOOK_DOWN);	/* diag dn-lft */
    captured += testline(x,y,LOOK_LEFT, LOOK_NEUT);	/* left */
    captured += testline(x,y,LOOK_LEFT, LOOK_UP);	/* diag up-lft */
    captured += testline(x,y,LOOK_NEUT, LOOK_UP);	/* up */
    captured += testline(x,y,LOOK_RIGHT,LOOK_UP);	/* diag up-rt */

    return(captured);
}


/* actually execute the move - in one direction */

cvtline(x,y,dx,dy)
char x,y;
int dx,dy;
{
char otherplay;
char flag;
char tmp;
char location;

    tmp = 0;

    if (player == PLAYER_WHITE)
        otherplay = PLAYER_BLACK;
    else
        otherplay = PLAYER_WHITE;

    flag = adjustxy(&x,&y,dx,dy);

    if (flag == 1)
        return(0);

    while ((occupies(x,y) == otherplay) && (flag == 0))
    {
        tmp++;
	location = (y<<3) | x;
	position[location] = player;
	changed[location]  = 1;
	flag = adjustxy(&x,&y,dx,dy);
    }

    if (flag == 1)
        return(0);

    if (occupies(x,y) == player)
        return(tmp);

    return(0);
}


/* actually execute the move - direction by direction */

domove(x,y)
char x,y;
{
char captured;
char location;

    captured = 0;

    location = (y<<3) + x;

    position[location] = player;
    changed[location] = 1;

    if (testline(x,y,LOOK_RIGHT,LOOK_NEUT) > 0)		/* right */
         cvtline(x,y,LOOK_RIGHT,LOOK_NEUT);

    if (testline(x,y,LOOK_RIGHT,LOOK_DOWN) > 0)		/* diag dn-rt */
         cvtline(x,y,LOOK_RIGHT,LOOK_DOWN);
    
    if (testline(x,y,LOOK_NEUT, LOOK_DOWN) > 0)		/* down */
         cvtline(x,y,LOOK_NEUT, LOOK_DOWN);

    if (testline(x,y,LOOK_LEFT, LOOK_DOWN) > 0)		/* diag dn-lft */
         cvtline(x,y,LOOK_LEFT, LOOK_DOWN);

    if (testline(x,y,LOOK_LEFT, LOOK_NEUT) > 0)		/* left */
         cvtline(x,y,LOOK_LEFT, LOOK_NEUT);

    if (testline(x,y,LOOK_LEFT, LOOK_UP) > 0)		/* diag up-lft */
         cvtline(x,y,LOOK_LEFT, LOOK_UP);

    if (testline(x,y,LOOK_NEUT, LOOK_UP) > 0)		/* up */
         cvtline(x,y,LOOK_NEUT, LOOK_UP);

    if (testline(x,y,LOOK_RIGHT,LOOK_UP) > 0)		/* diag up-rt */
         cvtline(x,y,LOOK_RIGHT,LOOK_UP);
}


computermove()
{
char i;

    maxprio = 0;
    for (i=0; i<64; i++)
    {
        if (priority[i] > maxprio)
	{
            maxprio = priority[i];
	    maxpos = i;
	}
    }

    if (maxprio > 0)
    {
        ptrx = maxpos & 7;
	ptry = maxpos >> 3;
    }

    return(maxprio);
}


/* execute the game from setup board until endgame */

playgame()
{
char i, move;
char location;

    cls();

    init_satb();

    set_sprpal(SPRPAL, colors);
    set_bgpal(BGPAL, colors);


/* set up playing surface */

    for (i=0; i<64; i++)
    {
        position[i] = 0;
	changed[i]  = 1;
    }
    position[(3<<3) + 3] = 1;
    position[(3<<3) + 4] = 2;
    position[(4<<3) + 3] = 2;
    position[(4<<3) + 4] = 1;

    displayall();

/* start with white's turn */

    player = PLAYER_WHITE;
    dispscore();

/* initial pointer position */

    ptrx = 3;
    ptry = 5;

/* verify # of good moves in each */
/* position, for current player   */

    setpriority();


/* MAIN LOOP */

    while (1) {
        vsync(0);
        satb_update();

	/* based on joystick delta, move pointer */

/*	move = joytrg(player);  */

	if ((computer == 1) && (player == PLAYER_BLACK))
	{
            selected = computermove();

            if (selected > 0)
            {
	        /* reposition pointer */
                ptr_move();
	        vsync(60);
                domove(ptrx, ptry);
            }
	}
	else
	{
	    move = joytrg(0);

	    if (move & JOY_LEFT)
	    {
                if (ptrx > 0)
                    ptrx--;
	    }
	    if (move & JOY_RGHT)
	    {
                if (ptrx < 7)
                    ptrx++;
	    }
	    if (move & JOY_UP)
	    {
                if (ptry > 0)
                    ptry--;
	    }
	    if (move & JOY_DOWN)
	    {
                if (ptry < 7)
                    ptry++;
	    }
	    if (move & JOY_A)
	    {
                location = (ptry<<3) | ptrx;
                selected = priority[location];
	        if (selected > 0)
                    domove(ptrx,ptry);
	    }
	    if (move & JOY_SLCT)
	    {
                if ( (joy(0) & (JOY_B|JOY_SLCT)) == (JOY_B|JOY_SLCT) )
	        {
                    disppriority();
                    while ( (joy(0) & (JOY_B|JOY_SLCT)) == (JOY_B|JOY_SLCT) );
                    displayall();
	        }
	    }
	}

	/* reposition pointer */

        ptr_move();

	/* if move taken, display squares that have flipped */

	if (selected != 0)
	{
	    selected = 0;

	    if (player == PLAYER_WHITE)
                player = PLAYER_BLACK;
	    else
                player = PLAYER_WHITE;

	    displaydirty();
	    dispscore();

            /* update # of good moves in each */
	    /* position, for current player   */

	    setpriority();

	    maxprio = 0;
	    for (i=0; i<64; i++)
	    {
                if (priority[i] > maxprio)
                    maxprio = priority[i];
	    }

	    if (maxprio == 0)
	    {
                /* player cannot make a move */
                /* switch player and check other player for end of game */

                if (player == PLAYER_WHITE)
                    player = PLAYER_BLACK;
                else
                    player = PLAYER_WHITE;

	        setpriority();

		maxprio = 0;
	        for (i=0; i<64; i++)
	        {
                    if (priority[i] > maxprio)
                        maxprio = priority[i];
	        }

		if (maxprio == 0)
		{
                /* end of game; determine winner */

		    player = 0;
		    dispscore();

                    put_string("     end of game", 0x146);

		    if (total1 > total2)
                        put_string("     white wins", 0x186);

                    else if (total1 < total2)
                        put_string("     black wins", 0x186);

                    else
                        put_string("     nobody wins", 0x186);

		    /* wait for return key */
		    while ((joytrg(0) & JOY_STRT) == 0);

                    /* hide the pointer */
                    spr_hide();


		    return(0);
		}
		else
		{

                    put_string("     cannot move  ", 0x106);
                    put_string("    press button a", 0x146);
                    put_string("     to continue  ", 0x186);
		    while ((joytrg(0) & JOY_A) == 0);
                    put_string("                  ", 0x106);
                    put_string("                  ", 0x146);
                    put_string("                  ", 0x186);
		}
	    }

	    displaydirty();
	    dispscore();
	}
    }
}


titlescreen()
{
char select;

    select = 0;

/* This will display the titlescreen soon */

    cls();

    put_string("flipit", 0x28C);
    put_string("by dave shadoff", 0x307);

    put_digit(2, 0x489);
    put_string("player", 0x48b);

    put_digit(1, 0x4c9);
    put_string("player vs computer", 0x4cb);

    while (1)
    {
        vsync(0);

	if ((joytrg(0) == JOY_DOWN) && (select == 0))
            select = 1;
	
	else if ((joytrg(0) == JOY_UP) && (select == 1))
            select = 0;
	
	else if (joytrg(0) == JOY_STRT)
        {
            if (select == 0)
                computer = 0;
	    else
                computer = 1;

	    return(0);
	}


        if (select == 0)
	{
            put_string("o", 0x487);
            put_string(" ", 0x4c7);
	}
	else
	{
            put_string(" ", 0x487);
            put_string("o", 0x4c7);
	}
    }

    ptrx = 8;
    ptry = 8;
    ptr_move();
}


main()
{

/* load some basic stuff */

    set_sprpal(SPRPAL, colors);
    set_bgpal(BGPAL, colors);
    computer = 0;   /* computer can't play yet */

/* load character data for squares */

    load_vram(VRAM_WHITE1, white1, 8);
    load_vram(VRAM_WHITE2, white2, 8);
    load_vram(VRAM_WHITE3, white3, 8);
    load_vram(VRAM_WHITE4, white4, 8);

    load_vram(VRAM_BLACK1, black1, 8);
    load_vram(VRAM_BLACK2, black2, 8);
    load_vram(VRAM_BLACK3, black3, 8);
    load_vram(VRAM_BLACK4, black4, 8);

    load_vram(VRAM_EMPTY1, empty1, 8);
    load_vram(VRAM_EMPTY2, empty2, 8);
    load_vram(VRAM_EMPTY3, empty3, 8);
    load_vram(VRAM_EMPTY4, empty4, 8);

/* load data for selection pointer */

    load_sprites(VRAM_SELECT, select, 1);

    while (1)
    {
        titlescreen();
        playgame();
    }
}

